<?php

namespace App\Http\Repositories\Eloquent\Admin;

use App\Models\Contact;
use App\Models\Page;
use App\Models\Partner;
use App\Models\Rating;
use App\Models\RequestModel;
use App\Models\Value;
use DevxPackage\AbstractRepository;
use Illuminate\Support\Facades\DB;

class PageRepository extends AbstractRepository
{

    protected $model;
    protected $partner;
    protected $rating;
    protected $value;
    protected $requestModel;
    protected $contact;

    public function __construct(Page $model, Partner $partner, Rating $rating, Value $value, RequestModel $requestModel, Contact $contact)
    {
        $this->model = $model;
        $this->partner = $partner;
        $this->rating = $rating;
        $this->value = $value;
        $this->requestModel = $requestModel;
        $this->contact = $contact;
    }

    public function crudName(): string
    {
        return 'pages';
    }

    public function screens($pageNo)
    {
        return $this->model->where('page_no', $pageNo)->first();
    }

    public function requests()
    {
        return $this->requestModel->with($this->model->model_relations())->withCount($this->model->model_relations_counts())->unArchive()->orderBy('id', 'DESC')->paginate(PAGINATION_COUNT);
    }

    public function contacts()
    {
        return $this->contact->unArchive()->orderBy('id', 'DESC')->paginate(PAGINATION_COUNT);
    }

    public function index($offset, $limit)
    {
        return $this->pagination($offset, $limit);
    }

    public function store($request)
    {
        DB::transaction(function () use ($request) {

            $page = $this->model->findOrFail($request->page_id);
            $data = $request->only(['title', 'info']);
            if ($request->hasFile('base_img')) {
                $data['base_img'] = uploadIamge($request->file('base_img'), $this->crudName()); // function on helper file to upload file
            }
            if ($request->hasFile('message_img')) {
                $data['message_img'] = uploadIamge($request->file('message_img'), $this->crudName()); // function on helper file to upload file
            }
            if ($request->hasFile('statistic_img')) {
                $data['statistic_img'] = uploadIamge($request->file('statistic_img'), $this->crudName()); // function on helper file to upload file
            }
            if (isset($request['statistics']) && count($request['statistics']) > 0) {
                $data['statistics'] = json_encode($request['statistics']);
            }
            $page->update($data);
            
            if ($request->hasFile('partner_logos')) {
                
                $rows = [];
                foreach ($request->file('partner_logos') as $file) {
                    $path = uploadIamge($file, $this->crudName()); // function on helper file to upload file
                    $rows[] = [
                        'img' => $path,
                        'page_id' => $page->id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }
                if ($rows) {
                    DB::table('partners')->insert($rows);
                }
            }
            if (isset($request->ratings) && count($request->ratings)) {
                
                $ratingRows = [];
                foreach ($request->ratings as $ratings) {
                    $path = null;
                    if (!empty($ratings['img'])) {
                        $path = uploadIamge($ratings['img'], $this->crudName()); // function on helper file to upload file
                    }
                    $ratingRows[] = [
                        'created_at' => now(),
                        'updated_at' => now(),
                        'page_id' => $page->id,
                        'img' => $path,
                        'title' => $ratings['title'],
                        'comment' => $ratings['comment'],
                        'rating_at' => date('Y-m-d', strtotime($ratings['rating_at'])),
                        'rating' => $ratings['rating'],
                    ];
                }
                if ($ratingRows) {
                    DB::table('ratings')->insert($ratingRows);
                }
            }
            if (isset($request->values) && count($request->values)) {
                
                foreach ($request->values as $value) {
                    $path = null;
                    $valueDB = new $this->value();
                    if (!empty($value['img'])) {
                        $path = uploadIamge($value['img'], $this->crudName()); // function on helper file to upload file
                    }
                    // dd($valueDB);
                    $valueDB->created_at = $valueDB->created_at ?? now();
                    $valueDB->updated_at = now();
                    $valueDB->page_id = $page->id;
                    $valueDB->title = $value['title'];
                    $valueDB->comment = $value['comment'];
                    if (isset($path) && !empty($path)) {
                        $valueDB->img = $path;
                    }
                    $valueDB->save();
                }
            }
        });
        flash()->success('Success');
        return back();
    }

    public function servicesUpdate($request, $id)
    {
        DB::transaction(function () use ($request, $id) {

            $valueDB = $this->value->where('id', $id)->first();
            if (isset($valueDB)) {
                $path = null;
                if (!empty($request['img'])) {
                    $path = uploadIamge($request['img'], $this->crudName()); // function on helper file to upload file
                }
                $valueDB->title = $request['title'];
                $valueDB->comment = $request['comment'];
                if (isset($path) && !empty($path)) {
                    $valueDB->img = $path;
                }
                $valueDB->save();
            }
        });
        flash()->success('Success');
        return back();
    }

    public function servicesDelete($id)
    {
        $valueDB = $this->value->where('id', $id)->first();
        if (isset($valueDB)) {
            $valueDB->delete();
        }
        flash()->success('Success');
        return back();
    }

    public function removePartners($id)
    {
        $this->partner->where('id', $id)->delete();
        flash()->success('Success');
        return back();
    }

    public function removeRating($id)
    {
        $this->rating->where('id', $id)->delete();
        flash()->success('Success');
        return back();
    }

    public function pagination($offset, $limit)
    {
        return $this->model->with($this->model->model_relations())->withCount($this->model->model_relations_counts())->unArchive()->orderBy('id', 'DESC')->paginate(PAGINATION_COUNT);
    }

    public function create(){}

    public function edit($id){}

    public function archivesPage($offset, $limit){}

}